<?php

namespace App\Http\Controllers;

use App\Models\Plato;
use App\Models\Carta;
use App\Models\SubCarta;
use App\Models\Alergeno;
use App\Models\Lang;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class PlatoController extends Controller
{
    public function index()
    {
        $cartas = Carta::with(['translations'])->activo()->ordenado()->get();
        $subcartas = SubCarta::with(['translations'])->activo()->ordenado()->get();
        $alergenos = Alergeno::with(['translations'])->activo()->ordenado()->get();
        /* $langs = Lang::where('activo', true)->get(); */
        $langs = Lang::whereNull('deleted_at')->get(); // solo idiomas activos
        return view('platos.index', compact('cartas', 'subcartas', 'alergenos', 'langs'));
    }

    public function ordenarVista()
    {
        $cartas = Carta::with([
            'translations',
            'subCartas.translations',
            'subCartas.platos.translations',
            'platos.translations',
        ])
            ->with([
                'subCartas.platos' => fn($q) => $q->orderBy('orden'),
                'platos' => fn($q) => $q->orderBy('orden')
            ])
            ->orderBy('orden')
            ->get();

        return view('platos.ordenar', compact('cartas'));
    }

    public function ordenarGuardar(Request $request)
    {
        $orden = $request->input('orden');
        // orden: { carta_id: { sub_id/null: [plato_ids...] } }

        foreach ($orden as $cartaId => $subGrupos) {
            foreach ($subGrupos as $subId => $platos) {
                foreach ($platos as $pos => $platoId) {
                    Plato::where('id', $platoId)->update(['orden' => $pos]);
                }
            }
        }

        return response()->json(['success' => true]);
    }

    public function byCarta(Carta $carta)
    {
        $platos = Plato::with(['translations', 'alergenos.translations', 'subCarta.translations'])
            ->where('carta_id', $carta->id)
            ->activo()
            ->ordenado()
            ->get();

        return response()->json([
            'success' => true,
            'carta' => $carta,
            'platos' => $platos
        ]);
    }

    public function show(Plato $plato)
    {
        $locale = app()->getLocale();

        $data = [
            'carta'       => $plato->carta?->translate($locale)->nombre,
            'subcarta'    => $plato->subCarta?->translate($locale)->nombre,
            'precio'      => $plato->precio,
            'orden'       => $plato->orden,
            'activo'      => $plato->activo,
            'alergenos'   => $plato->alergenos->map(fn($a) => $a->translate($locale)->nombre)->toArray(),
            'nombre'      => $plato->translate($locale)->nombre ?? null,
            'descripcion' => $plato->translate($locale)->descripcion ?? null,
            'imagen'      => $plato->imagen,
            'imagen_url'  => $plato->imagen ? asset($plato->imagen) : null,
        ];

        return response()->json($data);
    }

    public function datatables(Request $request)
    {
        try {
            $defaultLocale = Lang::where('default', true)->value('code') ?? 'es';

            $query = Plato::leftJoin('plato_langs', function ($join) use ($defaultLocale) {
                $join->on('platos.id', '=', 'plato_langs.plato_id')
                    ->where('plato_langs.locale', '=', $defaultLocale);
            })
                ->leftJoin('carta_langs', function ($join) use ($defaultLocale) {
                    $join->on('platos.carta_id', '=', 'carta_langs.carta_id')
                        ->where('carta_langs.locale', '=', $defaultLocale);
                })
                ->select(
                    'platos.*',
                    'plato_langs.nombre as nombre_traduccion',
                    'carta_langs.nombre as carta_nombre'
                );

            return DataTables::of($query)
                ->addColumn('imagen', function ($plato) {
                    if ($plato->imagen) {
                        return '<img src="' . $plato->imagen_url . '" width="50" height="50" class="rounded-circle">';
                    }
                    return '<div class="bg-light rounded-circle d-flex align-items-center justify-content-center" style="width:50px;height:50px;">
                                <i class="fas fa-utensils text-muted"></i>
                            </div>';
                })
                ->addColumn('nombre', function ($plato) {
                    return $plato->nombre_traduccion ?: '<span class="text-muted">Sin nombre</span>';
                })
                ->addColumn('carta', function ($plato) {
                    return $plato->carta_nombre ?: '<span class="text-muted">Sin carta</span>';
                })
                ->addColumn('precio_formateado', function ($plato) {
                    return '€' . number_format($plato->precio, 2);
                })
                ->editColumn('activo', function ($plato) {
                    return $plato->activo
                        ? '<span class="badge badge-success">Activo</span>'
                        : '<span class="badge badge-danger">Inactivo</span>';
                })
                ->addColumn('acciones', function ($plato) {
                    return '
                    <button class="btn btn-sm btn-info btn-ver" data-id="' . $plato->id . '">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-warning btn-editar" data-id="' . $plato->id . '">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger btn-eliminar" data-id="' . $plato->id . '">
                        <i class="fas fa-trash"></i>
                    </button>
                ';
                })
                ->rawColumns(['imagen', 'nombre', 'carta', 'precio_formateado', 'activo', 'acciones'])
                ->make(true);
        } catch (\Exception $e) {
            Log::error('Error en datatables: ' . $e->getMessage());
            return DataTables::of(Plato::query())->make(true);
        }
    }

    public function edit(Plato $plato)
    {
        $langs = \App\Models\Lang::pluck('code')->toArray();

        $translations = [];
        foreach ($langs as $locale) {
            $translations[$locale] = [
                'nombre'      => $plato->translate($locale)?->nombre ?? '',
                'descripcion' => $plato->translate($locale)?->descripcion ?? '',
            ];
        }

        return response()->json([
            'id'          => $plato->id,
            'carta_id'    => $plato->carta_id,
            'sub_carta_id' => $plato->sub_carta_id,
            'precio'      => $plato->precio,
            'orden'       => $plato->orden,
            'activo'      => $plato->activo,
            'imagen'      => $plato->imagen ? asset($plato->imagen) : null,
            'alergenos'   => $plato->alergenos->map(fn($a) => ['id' => $a->id])->toArray(),
            'translations' => $translations,
        ]);
    }

    /*     public function edit(Plato $plato)
    {
        $plato->load('translations', 'alergenos');
        return response()->json($plato);
    } */

    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $data = $request->validate([
                'plato_id' => 'nullable|exists:platos,id',
                'carta_id' => 'required|exists:cartas,id',
                'sub_carta_id' => 'nullable|exists:sub_cartas,id',
                'precio' => 'required|numeric|min:0',
                'activo' => 'nullable|boolean',
                'imagen' => 'nullable|image|max:2048',
                'alergenos' => 'nullable|array',
                'alergenos.*' => 'exists:alergenos,id',
                'nombre' => 'required|array',
                'descripcion' => 'nullable|array',
            ]);

            $plato = $request->plato_id
                ? Plato::findOrFail($request->plato_id)
                : new Plato();

            $plato->fill([
                'carta_id' => $data['carta_id'],
                'sub_carta_id' => $data['sub_carta_id'] ?? null,
                'precio' => $data['precio'],
                'orden' => 0,
                'activo' => $request->boolean('activo'),
            ]);

            if ($request->hasFile('imagen')) {
                // Eliminar imagen anterior si existe
                if ($plato->imagen && file_exists(public_path($plato->imagen))) {
                    unlink(public_path($plato->imagen));
                }

                // Guardar nueva imagen
                $file = $request->file('imagen');
                $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('images/platos'), $filename);

                // Guardamos la ruta relativa (para usar en <img src="{{ asset($plato->imagen) }}"> )
                $plato->imagen = 'images/platos/' . $filename;
            }

            if ($request->filled('eliminar_imagen')) {
                if ($plato->imagen && file_exists(public_path($plato->imagen))) {
                    unlink(public_path($plato->imagen));
                }
                $plato->imagen = null;
            }


            $plato->save();

            foreach ($data['nombre'] as $locale => $nombre) {
                $plato->translateOrNew($locale)->nombre = $nombre;
                $plato->translateOrNew($locale)->descripcion = $data['descripcion'][$locale] ?? null;
                /* $plato->translateOrNew($locale)->slug = \Str::slug($nombre); */
            }
            $plato->save();

            $plato->alergenos()->sync($data['alergenos'] ?? []);

            DB::commit();

            return response()->json(['success' => true, 'message' => $request->plato_id ? 'Plato actualizado' : 'Plato creado']);
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $th->getMessage()], 500);
        }
    }

    public function destroy(Plato $plato)
    {
        if ($plato->imagen && file_exists(public_path($plato->imagen))) {
            unlink(public_path($plato->imagen));
        }

        $plato->delete();
        return response()->json(['success' => true, 'message' => 'Plato eliminado']);
    }

    /*  public function show(Plato $plato)
    {
        $plato->load('translations', 'alergenos');
        return view('admin.platos.show', compact('plato'));
    } */
}
