<?php

namespace App\Http\Controllers;

use App\Models\Carta;
use App\Models\Lang;
use App\Models\SubCarta;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class CartaController extends Controller
{
    public function index()
    {
        return view('cartas.index');
    }

    public function ordenarVista()
    {
        $cartas = Carta::withTranslation()
            ->orderBy('orden')
            ->get();

        return view('cartas.ordenar', compact('cartas'));
    }

    public function ordenarGuardar(Request $request)
    {
        $orden = $request->input('orden'); // array de IDs en el nuevo orden

        foreach ($orden as $pos => $id) {
            Carta::where('id', $id)->update(['orden' => $pos]);
        }

        return response()->json(['success' => true]);
    }
    
    public function getSubCartas($cartaId)
    {
        $subCartas = SubCarta::where('carta_id', $cartaId)
            ->where('activo', true)
            ->orderBy('orden')
            ->withTranslation() // 👈 carga la traducción del locale actual
            ->get()
            ->map(fn($sc) => [
                'id' => $sc->id,
                'nombre' => $sc->nombre, // ya trae traducción automática
            ]);

        return response()->json($subCartas);
    }


    public function data()
    {
        $cartas = Carta::with('translations')->select('cartas.*');
        return DataTables::of($cartas)
            ->addColumn('nombre', function ($carta) {
                $default = Lang::default()->code;
                return $carta->translate($default)->nombre ?? '';
            })
            ->addColumn('activo', fn($carta) => $carta->activo ? 'Sí' : 'No')
            ->addColumn('acciones', function ($carta) {
                return '
                    <button class="btn btn-sm btn-primary edit" data-id="' . $carta->id . '">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger delete" data-id="' . $carta->id . '">
                        <i class="fas fa-trash-alt"></i>
                    </button>
                ';
            })
            ->rawColumns(['acciones'])
            ->make(true);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'orden' => 'nullable|integer',
            'activo' => 'boolean',
            'nombre' => 'required|array',
            'descripcion' => 'nullable|array',
        ]);

        $carta = $request->id ? Carta::find($request->id) : new Carta();
        $carta->orden = $request->orden ?? 0;
        $carta->activo = $request->activo ?? true;
        $carta->save();

        // Guardar traducciones dinámicamente
        foreach ($request->nombre as $locale => $nombre) {
            $carta->translateOrNew($locale)->nombre = $nombre;
            $carta->translateOrNew($locale)->descripcion = $request->descripcion[$locale] ?? null;
        }

        $carta->save();

        return response()->json(['success' => true]);
    }

    public function edit(Carta $carta)
    {
        $carta->load('translations');
        $data = [
            'id' => $carta->id,
            'orden' => $carta->orden,
            'activo' => $carta->activo,
            'nombre' => [],
            'descripcion' => [],
        ];

        foreach ($carta->translations as $t) {
            $data['nombre'][$t->locale] = $t->nombre;
            $data['descripcion'][$t->locale] = $t->descripcion;
        }

        return response()->json($data);
    }

    public function destroy(Carta $carta)
    {
        $carta->delete(); // soft delete
        return response()->json(['success' => true]);
    }
}
