<?php

namespace App\Http\Controllers;

use App\Models\Bebida;
use App\Models\Carta;
use App\Models\SubCarta;
use App\Models\Alergeno;
use App\Models\BebidaPrecio;
use App\Models\Lang;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class BebidaController extends Controller
{
    public function index()
    {
        $cartas = Carta::with(['translations'])->activo()->ordenado()->get();
        $subcartas = SubCarta::with(['translations'])->activo()->ordenado()->get();
        /* $alergenos = Alergeno::with(['translations'])->activo()->ordenado()->get(); */
        /* $langs = Lang::where('activo', true)->get(); */
        $langs = Lang::whereNull('deleted_at')->get(); // solo idiomas activos
        return view('bebidas.index', compact('cartas', 'subcartas', 'langs'));
    }

    public function ordenarVista()
    {
        $cartas = Carta::with([
            'translations',
            'bebidas.translations',
            'subCartas.translations',
            'subCartas.bebidas.translations',
        ])
            ->with([
                'bebidas' => fn($q) => $q->orderBy('orden'),
                'subCartas.bebidas' => fn($q) => $q->orderBy('orden'),
            ])
            ->orderBy('orden')
            ->get();

        return view('bebidas.ordenar', compact('cartas'));
    }

    public function ordenarGuardar(Request $request)
    {
        $orden = $request->input('orden');

        foreach ($orden as $cartaId => $subGrupos) {
            foreach ($subGrupos as $subId => $bebidas) {
                foreach ($bebidas as $pos => $bebidaId) {
                    Bebida::where('id', $bebidaId)->update(['orden' => $pos]);
                }
            }
        }

        return response()->json(['success' => true]);
    }
    
    public function byCarta(Carta $carta)
    {
        $bebidas = Bebida::with(['translations', 'subCarta.translations', 'precios'])
            ->where('carta_id', $carta->id)
            ->activo()
            ->ordenado()
            ->get()
            ->map(function ($bebida) {
                return [
                    'id'          => $bebida->id,
                    'nombre'      => $bebida->nombre,
                    'descripcion' => $bebida->descripcion,
                    'imagen'      => $bebida->imagen ? $bebida->imagen : null,
                    'sub_carta'   => $bebida->subCarta ? $bebida->subCarta->nombre : null,
                    'precios'     => $bebida->precios->map(function ($precio) {
                        return [
                            'tipo_presentacion' => $precio->tipo_presentacion,
                            'precio'            => (float) $precio->precio,
                        ];
                    })->values(),
                ];
            });

        return response()->json([
            'success' => true,
            'carta'   => [
                'id'     => $carta->id,
                'orden'  => $carta->orden,
                'activo' => $carta->activo,
            ],
            'bebidas' => $bebidas
        ]);
    }

    /*     public function byCarta(Carta $carta)
    {
        $bebidas = Bebida::with(['translations', 'subCarta.translations'])
            ->where('carta_id', $carta->id)
            ->activo()
            ->ordenado()
            ->get();


        return response()->json([
            'success' => true,
            'carta' => $carta,
            'bebidas' => $bebidas
        ]);
    } */

    public function show(Bebida $bebida)
    {
        $locale = app()->getLocale();

        $data = [
            'carta'       => $bebida->carta?->translate($locale)->nombre,
            'subcarta'    => $bebida->subCarta?->translate($locale)->nombre,
            'precio'      => $bebida->precio,
            'orden'       => $bebida->orden,
            'activo'      => $bebida->activo,
            'alergenos'   => $bebida->alergenos->map(fn($a) => $a->translate($locale)->nombre)->toArray(),
            'nombre'      => $bebida->translate($locale)->nombre ?? null,
            'descripcion' => $bebida->translate($locale)->descripcion ?? null,
            'imagen'      => $bebida->imagen,
            'imagen_url'  => $bebida->imagen ? asset($bebida->imagen) : null,
        ];

        return response()->json($data);
    }

    public function datatables(Request $request)
    {
        try {
            $defaultLocale = Lang::where('default', true)->value('code') ?? 'es';

            $query = Bebida::leftJoin('bebida_langs', function ($join) use ($defaultLocale) {
                $join->on('bebidas.id', '=', 'bebida_langs.bebida_id')
                    ->where('bebida_langs.locale', '=', $defaultLocale);
            })
                ->leftJoin('carta_langs', function ($join) use ($defaultLocale) {
                    $join->on('bebidas.carta_id', '=', 'carta_langs.carta_id')
                        ->where('carta_langs.locale', '=', $defaultLocale);
                })
                ->select(
                    'bebidas.*',
                    'bebida_langs.nombre as nombre_traduccion',
                    'carta_langs.nombre as carta_nombre'
                );

            return DataTables::of($query)
                ->addColumn('imagen', function ($bebida) {
                    if ($bebida->imagen) {
                        return '<img src="' . $bebida->imagen_url . '" width="50" height="50" class="rounded-circle">';
                    }
                    return '<div class="bg-light rounded-circle d-flex align-items-center justify-content-center" style="width:50px;height:50px;">
                                <i class="fas fa-utensils text-muted"></i>
                            </div>';
                })
                ->addColumn('nombre', function ($bebida) {
                    return $bebida->nombre_traduccion ?: '<span class="text-muted">Sin nombre</span>';
                })
                ->addColumn('carta', function ($bebida) {
                    return $bebida->carta_nombre ?: '<span class="text-muted">Sin carta</span>';
                })
                ->addColumn('precio_formateado', function ($bebida) {
                    return '€' . number_format($bebida->precio, 2);
                })
                ->editColumn('activo', function ($bebida) {
                    return $bebida->activo
                        ? '<span class="badge badge-success">Activo</span>'
                        : '<span class="badge badge-danger">Inactivo</span>';
                })
                ->addColumn('acciones', function ($bebida) {
                    return '
                    <button class="btn btn-sm btn-info btn-ver" data-id="' . $bebida->id . '">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-warning btn-editar" data-id="' . $bebida->id . '">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger btn-eliminar" data-id="' . $bebida->id . '">
                        <i class="fas fa-trash"></i>
                    </button>
                ';
                })
                ->rawColumns(['imagen', 'nombre', 'carta', 'precio_formateado', 'activo', 'acciones'])
                ->make(true);
        } catch (\Exception $e) {
            Log::error('Error en datatables: ' . $e->getMessage());
            return DataTables::of(Bebida::query())->make(true);
        }
    }

    public function edit(Bebida $bebida)
    {
        $langs = Lang::pluck('code')->toArray();

        $translations = [];
        foreach ($langs as $locale) {
            $translations[$locale] = [
                'nombre'      => $bebida->translate($locale)?->nombre ?? '',
                'descripcion' => $bebida->translate($locale)?->descripcion ?? '',
            ];
        }

        // Precios por presentación
        $precios = [
            'copa'     => $bebida->precios()->where('tipo_presentacion', 'copa')->value('precio') ?? 0,
            'vaso'     => $bebida->precios()->where('tipo_presentacion', 'vaso')->value('precio') ?? 0,
            'botella'  => $bebida->precios()->where('tipo_presentacion', 'botella')->value('precio') ?? 0,
            'chupito'  => $bebida->precios()->where('tipo_presentacion', 'chupito')->value('precio') ?? 0,
        ];

        return response()->json([
            'id'           => $bebida->id,
            'carta_id'     => $bebida->carta_id,
            'sub_carta_id' => $bebida->sub_carta_id,
            'orden'        => $bebida->orden,
            'activo'       => $bebida->activo,
            'imagen'       => $bebida->imagen ? asset($bebida->imagen) : null,
            'translations' => $translations,
            'precios'      => $precios, // 👈 agregado
        ]);
    }


    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $data = $request->validate([
                'bebida_id'     => 'nullable|exists:bebidas,id',
                'carta_id'      => 'required|exists:cartas,id',
                'sub_carta_id'  => 'nullable|exists:sub_cartas,id',
                'orden'         => 'nullable',
                'activo'        => 'nullable|boolean',
                'imagen'        => 'nullable|image|max:2048',
                'nombre'        => 'required|array',   // 👈 nombres en distintos idiomas
                'descripcion'   => 'nullable|array',   // 👈 descripciones en distintos idiomas
                'precio_copa'   => 'nullable|numeric|min:0',
                'precio_vaso'   => 'nullable|numeric|min:0',
                'precio_botella' => 'nullable|numeric|min:0',
                'precio_chupito' => 'nullable|numeric|min:0',
            ]);

            $bebida = $request->bebida_id
                ? Bebida::findOrFail($request->bebida_id)
                : new Bebida();

            $bebida->fill([
                'carta_id'     => $data['carta_id'],
                'sub_carta_id' => $data['sub_carta_id'] ?? null,
                'orden'        => 0,
                'activo'       => $request->boolean('activo'),
            ]);

            // Manejo de imagen
            if ($request->hasFile('imagen')) {
                if ($bebida->imagen && file_exists(public_path($bebida->imagen))) {
                    unlink(public_path($bebida->imagen));
                }
                $file = $request->file('imagen');
                $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('images/bebidas'), $filename);
                $bebida->imagen = 'images/bebidas/' . $filename;
            }

            if ($request->filled('eliminar_imagen')) {
                if ($bebida->imagen && file_exists(public_path($bebida->imagen))) {
                    unlink(public_path($bebida->imagen));
                }
                $bebida->imagen = null;
            }

            $bebida->save();

            // ✅ Guardar traducciones
            foreach ($request->nombre as $locale => $nombre) {
                $bebida->translateOrNew($locale)->nombre = $nombre;
                $bebida->translateOrNew($locale)->descripcion = $request->descripcion[$locale] ?? null;
            }
            $bebida->save();

            // ✅ Guardar precios (updateOrCreate evita duplicados)
            $precios = [
                'copa'    => $request->precio_copa,
                'vaso'    => $request->precio_vaso,
                'botella' => $request->precio_botella,
                'chupito' => $request->precio_chupito,
            ];

            foreach ($precios as $tipo => $precio) {
                if ($precio !== null && $precio !== '') {
                    BebidaPrecio::updateOrCreate(
                        ['bebida_id' => $bebida->id, 'tipo_presentacion' => $tipo],
                        ['precio' => $precio]
                    );
                } else {
                    BebidaPrecio::where('bebida_id', $bebida->id)
                        ->where('tipo_presentacion', $tipo)
                        ->delete();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $request->bebida_id ? 'Bebida actualizada' : 'Bebida creada'
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $th->getMessage()
            ], 500);
        }
    }


    public function destroy(Bebida $bebida)
    {
        if ($bebida->imagen && file_exists(public_path($bebida->imagen))) {
            unlink(public_path($bebida->imagen));
        }

        $bebida->delete();
        return response()->json(['success' => true, 'message' => 'Bebida eliminado']);
    }

    /*  public function show(Bebida $bebida)
    {
        $bebida->load('translations', 'alergenos');
        return view('admin.bebidas.show', compact('bebida'));
    } */
}
