<?php

namespace App\Http\Controllers;

use App\Models\Alergeno;
use App\Models\Lang;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AlergenoController extends Controller
{
    public function index()
    {
        $langs = Lang::whereNull('deleted_at')->get(); // solo idiomas activos
        return view('alergenos.index', compact('langs'));
    }

    public function ordenarVista()
    {
        $alergenos = Alergeno::withTranslation()
            ->orderBy('orden')
            ->get();

        return view('alergenos.ordenar', compact('alergenos'));
    }

    public function ordenarGuardar(Request $request)
    {
        $orden = $request->input('orden'); // array de IDs

        foreach ($orden as $pos => $id) {
            Alergeno::where('id', $id)->update(['orden' => $pos]);
        }

        return response()->json(['success' => true]);
    }
    
    public function datatables(Request $request)
    {
        try {
            // Obtener el código del idioma por defecto
            $defaultLocale = Lang::where('default', true)->value('code');

            // Si no hay idioma por defecto, usar el primero activo
            if (!$defaultLocale) {
                $defaultLocale = Lang::where('activo', true)->value('code') ?? 'es';
            }

            $query = Alergeno::leftJoin('alergeno_langs', function ($join) use ($defaultLocale) {
                $join->on('alergenos.id', '=', 'alergeno_langs.alergeno_id')
                    ->where('alergeno_langs.locale', '=', $defaultLocale);
            })
                ->select('alergenos.*', 'alergeno_langs.nombre as nombre_traduccion');

            return DataTables::of($query)
                ->addColumn('nombre', function ($alergeno) {
                    return $alergeno->nombre_traduccion ?: '<span class="text-muted">Sin nombre</span>';
                })
                ->editColumn('activo', function ($alergeno) {
                    return $alergeno->activo
                        ? '<span class="badge badge-success">Activo</span>'
                        : '<span class="badge badge-danger">Inactivo</span>';
                })
                ->addColumn('acciones', function ($alergeno) {
                    return '
                    <button class="btn btn-sm btn-warning btn-editar" data-id="' . $alergeno->id . '">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger btn-eliminar" data-id="' . $alergeno->id . '">
                        <i class="fas fa-trash"></i>
                    </button>
                ';
                })
                ->filterColumn('nombre', function ($query, $keyword) use ($defaultLocale) {
                    $query->where('alergeno_langs.nombre', 'like', "%{$keyword}%");
                })
                ->orderColumn('nombre', 'alergeno_langs.nombre $1')
                ->rawColumns(['nombre', 'activo', 'acciones'])
                ->make(true);
        } catch (\Exception $e) {
            // Log del error
            Log::error('Error en datatables: ' . $e->getMessage());

            // Devolver una respuesta vacía para DataTables
            return DataTables::of(Alergeno::query())->make(true);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'orden' => 'required|integer',
            'activo' => 'boolean',
            'nombre.*' => 'required|string|max:255',
            'descripcion.*' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $alergeno = new Alergeno();
            $alergeno->orden = $request->orden;
            $alergeno->activo = $request->activo ?? true;
            $alergeno->save();

            // Guardar traducciones
            foreach ($request->nombre as $locale => $nombre) {
                $alergeno->translateOrNew($locale)->nombre = $nombre;
                $alergeno->translateOrNew($locale)->descripcion = $request->descripcion[$locale] ?? null;
            }

            $alergeno->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Alérgeno creado correctamente'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al crear el alérgeno: ' . $e->getMessage()
            ], 500);
        }
    }
    /* 
    public function show(Alergeno $alergeno)
    {
        $alergeno->load('translations');
        return response()->json($alergeno);
    } */
    public function show(Alergeno $alergeno)
    {
        try {
            // Cargar las traducciones de manera eficiente
            $translations = $alergeno->translations->map(function ($translation) {
                return [
                    'locale' => $translation->locale,
                    'nombre' => $translation->nombre,
                    'descripcion' => $translation->descripcion
                ];
            });

            return response()->json([
                'success' => true,
                'id' => $alergeno->id,
                'orden' => $alergeno->orden,
                'activo' => (bool)$alergeno->activo,
                'translations' => $translations
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar el alérgeno: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, Alergeno $alergeno)
    {
        // DEBUG: Ver qué está llegando al controlador
        /* dd($request->all()); */
        $validator = Validator::make($request->all(), [
            'orden' => 'required|integer',
            'activo' => 'boolean',
            'nombre.*' => 'required|string|max:255',
            'descripcion.*' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $alergeno->orden = $request->orden;
            $alergeno->activo = $request->activo ?? true;

            // Actualizar traducciones
            foreach ($request->nombre as $locale => $nombre) {
                $alergeno->translateOrNew($locale)->nombre = $nombre;
                $alergeno->translateOrNew($locale)->descripcion = $request->descripcion[$locale] ?? null;
            }
            $alergeno->activo = $request->activo ?? true;
            $alergeno->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Alérgeno actualizado correctamente'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el alérgeno: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Alergeno $alergeno)
    {
        try {
            $alergeno->delete();
            return response()->json([
                'success' => true,
                'message' => 'Alérgeno eliminado correctamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el alérgeno: ' . $e->getMessage()
            ], 500);
        }
    }
}
